// license:BSD-3-Clause
// copyright-holders:hap
// thanks-to:Berger
/******************************************************************************

Novag Supremo

Hardware notes:
- Hitachi HD63A03YP MCU @ 8MHz (2MHz internal)
- 32KB ROM(TC57256AD-12), 2KB RAM(TC5516APL)
- LCD with 4 digits and custom segments, no LCD chip
- buzzer, 16 LEDs, 8*8 chessboard buttons

Novag Primo is assumed to be on similar hardware
Supremo also had a "limited edition" rerelease in 1990, plastic is fake-wood
instead of black, otherwise it's the same game.

TODO:
- does not work, most likely due to incomplete cpu emulation (unemulated timer registers),
  could also be a bad rom dump on top of that - even when adding IRQ3 with a hack, it
  doesn't do much at all
- I/O seems very similar to nsnova, can drivers be merged? (get this one working first)
- is 1988 version the same ROM?

******************************************************************************/

#include "emu.h"

#include "cpu/m6800/m6801.h"
#include "machine/sensorboard.h"
#include "sound/dac.h"

#include "speaker.h"


namespace {

class supremo_state : public driver_device
{
public:
	supremo_state(const machine_config &mconfig, device_type type, const char *tag) :
		driver_device(mconfig, type, tag),
		m_maincpu(*this, "maincpu"),
		m_board(*this, "board"),
		m_dac(*this, "dac")
	{ }

	// machine configs
	void supremo(machine_config &config);

protected:
	virtual void machine_start() override;

private:
	// devices/pointers
	required_device<hd6303y_cpu_device> m_maincpu;
	required_device<sensorboard_device> m_board;
	required_device<dac_bit_interface> m_dac;

	void main_map(address_map &map);

	u8 p2_r();
	void p2_w(u8 data);
	void p5_w(u8 data);
	void p6_w(u8 data);
};

void supremo_state::machine_start()
{
}



/******************************************************************************
    I/O
******************************************************************************/


u8 supremo_state::p2_r()
{
	return 1;
}

void supremo_state::p2_w(u8 data)
{
	// P22: speaker out
	m_dac->write(BIT(data, 2));
}

void supremo_state::p5_w(u8 data)
{
}

void supremo_state::p6_w(u8 data)
{
}



/******************************************************************************
    Address Maps
******************************************************************************/

void supremo_state::main_map(address_map &map)
{
	map(0x0000, 0x0027).m(m_maincpu, FUNC(hd6303y_cpu_device::hd6301y_io));
	map(0x0040, 0x013f).ram(); // internal
	map(0x4000, 0x47ff).ram();
	map(0x8000, 0xffff).rom();
}



/******************************************************************************
    Input Ports
******************************************************************************/

static INPUT_PORTS_START( supremo )
INPUT_PORTS_END



/******************************************************************************
    Machine Configs
******************************************************************************/

void supremo_state::supremo(machine_config &config)
{
	/* basic machine hardware */
	HD6303Y(config, m_maincpu, 8_MHz_XTAL);
	m_maincpu->set_addrmap(AS_PROGRAM, &supremo_state::main_map);
	m_maincpu->in_p2_cb().set(FUNC(supremo_state::p2_r));
	m_maincpu->out_p2_cb().set(FUNC(supremo_state::p2_w));
	m_maincpu->out_p5_cb().set(FUNC(supremo_state::p5_w));
	m_maincpu->out_p6_cb().set(FUNC(supremo_state::p6_w));

	// THIS IS A HACK, vector @ 0xffec, use ROM_COPY
	//const attotime irq_period = attotime::from_ticks(4 * 128 * 10, 8_MHz_XTAL);
	//m_maincpu->set_periodic_int(FUNC(supremo_state::irq0_line_hold), irq_period);

	SENSORBOARD(config, m_board).set_type(sensorboard_device::BUTTONS);
	m_board->init_cb().set(m_board, FUNC(sensorboard_device::preset_chess));
	m_board->set_delay(attotime::from_msec(150));

	/* sound hardware */
	SPEAKER(config, "speaker").front_center();
	DAC_1BIT(config, m_dac).add_route(ALL_OUTPUTS, "speaker", 0.25);
}



/******************************************************************************
    ROM Definitions
******************************************************************************/

ROM_START( supremo )
	ROM_REGION( 0x10000, "maincpu", 0 )
	ROM_LOAD("sp_a10.u5", 0x8000, 0x8000, BAD_DUMP CRC(745d010f) SHA1(365a8e2afcf63678ba0161b9082f6439a9d78c9f) )
ROM_END

} // anonymous namespace



/******************************************************************************
    Drivers
******************************************************************************/

//    YEAR  NAME     PARENT  COMPAT  MACHINE  INPUT     CLASS          INIT        COMPANY, FULLNAME, FLAGS
CONS( 1990, supremo, 0,      0,      supremo,  supremo, supremo_state, empty_init, "Novag", "Supremo - Limited Edition", MACHINE_SUPPORTS_SAVE | MACHINE_NOT_WORKING )

